if ('serviceWorker' in navigator) {
  window.addEventListener('load', () => {
    navigator.serviceWorker.register('sw.js')
      .then((registration) => {
        console.log('ServiceWorker registration successful with scope: ', registration.scope);
      })
      .catch((err) => {
        console.log('ServiceWorker registration failed: ', err);
      });
  });
}

// PWA Install Prompt Logic
let deferredPrompt;

// Check if it's iOS
const isIos = () => {
  const userAgent = window.navigator.userAgent.toLowerCase();
  return /iphone|ipad|ipod/.test(userAgent);
}

// Check if app is in standalone mode
const isInStandaloneMode = () => {
  return ('standalone' in window.navigator) && (window.navigator.standalone);
}

window.addEventListener('beforeinstallprompt', (e) => {
  // Prevent the mini-infobar from appearing on mobile
  e.preventDefault();
  // Stash the event so it can be triggered later.
  deferredPrompt = e;
  // Update UI notify the user they can install the PWA
  showInstallPromotion();
  console.log('beforeinstallprompt fired');
});

function showInstallPromotion() {
  const installBtn = document.getElementById('pwa-install-btn');
  if (installBtn) {
    installBtn.style.display = 'inline-flex'; // Show the button
    installBtn.addEventListener('click', async () => {
      // Hide the app provided install promotion
      installBtn.style.display = 'none';
      // Show the install prompt
      deferredPrompt.prompt();
      // Wait for the user to respond to the prompt
      const { outcome } = await deferredPrompt.userChoice;
      console.log(`User response to the install prompt: ${outcome}`);
      // We've used the prompt, and can't use it again, throw it away
      deferredPrompt = null;
    });
  }
}

window.addEventListener('appinstalled', () => {
  // Hide the app-provided install promotion
  const installBtn = document.getElementById('pwa-install-btn');
  if (installBtn) {
    installBtn.style.display = 'none';
  }
  // Clear the deferredPrompt so it can be garbage collected
  deferredPrompt = null;
  console.log('PWA was installed');
});

// iOS specific logic
if (isIos() && !isInStandaloneMode()) {
    // We only show this once per session or use local storage to show it less frequently
    if (!sessionStorage.getItem('iosInstallPromptShown')) {
        showIosInstallPrompt();
    }
}

function showIosInstallPrompt() {
    const modal = document.createElement('div');
    modal.className = 'ios-install-modal';
    modal.innerHTML = `
        <div class="ios-install-header">
            <span class="ios-install-title">Install UANG KAS</span>
            <button class="ios-install-close">&times;</button>
        </div>
        <div class="ios-install-steps">
            <div class="ios-step">
                <i class="fas fa-share-square ios-share-icon"></i>
                <span>Tap tombol <strong>Share</strong> di bawah</span>
            </div>
            <div class="ios-step">
                <i class="fas fa-plus-square"></i>
                <span>Pilih <strong>Add to Home Screen</strong></span>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Show modal
    setTimeout(() => {
        modal.style.display = 'block';
    }, 1000);
    
    // Handle close
    const closeBtn = modal.querySelector('.ios-install-close');
    closeBtn.addEventListener('click', () => {
        modal.style.display = 'none';
        sessionStorage.setItem('iosInstallPromptShown', 'true');
    });
}
